<?php

namespace Motta\Addons\Elementor\Builder;

use WooCommerce\PayPalCommerce\WcGateway\Endpoint\ReturnUrlEndpoint;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Main class of plugin for admin
 */
class Settings  {

	/**
	 * Instance
	 *
	 * @var $instance
	 */
	private static $instance;


	/**
	 * Initiator
	 *
	 * @since 1.0.0
	 * @return object
	 */
	public static function instance() {
		if ( ! isset( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	const POST_TYPE     = 'motta_builder';
	const OPTION_NAME   = 'motta_builder';
	const TAXONOMY_TYPE     = 'motta_builder_type';
	/**
	 * Instantiate the object.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function __construct() {
		add_action( 'motta_register_theme_settings', array( $this, 'register_motta_builder_settings' ) );

		// Handle post columns
		add_filter( sprintf( 'manage_%s_posts_columns', self::POST_TYPE ), array( $this, 'edit_admin_columns' ) );
		add_action( sprintf( 'manage_%s_posts_custom_column', self::POST_TYPE ), array( $this, 'manage_custom_columns' ), 10, 2 );

		add_action( 'wp_ajax_motta_save_builder_enable', array( $this, 'save_builder_enable' ) );

		add_action( 'wp_ajax_motta_builder_template_type', array( $this, 'builder_template_type' ) );

		// Add custom post type to screen ids
		add_filter('woocommerce_screen_ids', array( $this, 'wc_screen_ids' ) );

		add_filter( 'views_edit-' . self::POST_TYPE, array($this, 'admin_print_tabs' ) );

		add_filter( 'parse_query', array( $this, 'query_filter' ) );

		// Enqueue style and javascript
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts' ) );

		// Add meta boxes.
		add_action( 'save_post', array( $this, 'delete_builder_cache' ), 10, 1 );
		add_action( 'wp_trash_post', array( $this, 'delete_builder_cache' ), 10, 1 );
		add_action( 'before_delete_post', array( $this, 'delete_builder_cache' ), 10, 1 );
		add_action( 'motta_after_builder_enable', array( $this, 'delete_builder_cache' ), 10, 1 );

		add_action( 'admin_footer', array( $this, 'template_type_popup' ) );
	}

	public function register_motta_builder_settings( $columns ) {
		add_settings_section(
			'motta_builder_section',
			'',
			array($this, 'motta_builder_section_html'),
			'theme_settings'
		);
		register_setting(
			'theme_settings',
			'motta_builder_enable',
			'string'
		);
		add_settings_field(
			'motta_builder_enable',
			esc_html__( 'Motta Builder Enable', 'motta-addons' ),
			array($this,'motta_builder_enable_field_html'),
			'theme_settings',
			'motta_builder_section'
		);
	}

	public function motta_builder_enable_field_html() {
		$default = 1;
		$enable = get_option( 'motta_builder_enable' );
		$disable = '';
		$notice = '';
		if (class_exists('\Elementor\Plugin') && ! \Elementor\Plugin::$instance->experiments->is_feature_active( 'container' ) ) {
			$disable = ! $enable ? 'disabled' : '';
			$notice = sprintf('<p>%s</p>', esc_html('In order to use Motta Builder, first you need to active Grid Container in Elementor. Go to Elementor > Settings > Features > Container to select active option.', 'motta-addons'));
			$notice .= sprintf('<a href="%s">%s</a>', esc_url(admin_url('admin.php?page=elementor-settings#tab-experiments')), esc_html('Active Elementor Container', 'motta-addons'));
		}
		?>
		<input id="motta_builder_enable_checkbox" type="checkbox" name="motta_builder_enable" <?php checked( $default, $enable ); ?> <?php echo esc_attr( $disable ); ?> value="<?php echo $default; ?>">
		<label for="motta_builder_enable_checkbox"><?php esc_html_e('Yes', 'motta-addons'); ?></label>
		<?php
		echo $notice;
		?>
		<?php
	}

	public function motta_builder_section_html() {
		echo '<hr/><h2>' . esc_html__('Motta Builder', 'motta-addons') . '</h2>';
	}

	/**
	 * Add custom column to builders management screen
	 * Add Thumbnail column
     *
	 * @since 1.0.0
	 *
	 * @param  array $columns Default columns
	 *
	 * @return array
	 */
	public function edit_admin_columns( $columns ) {
		$columns = array_merge( $columns, array(
			'builder_type' => esc_html__( 'Type', 'motta-addons' ),
			'enable_builder' => esc_html__( 'Default Builder', 'motta-addons' ),
		) );

		return $columns;
	}

	/**
	 * Handle custom column display
     *
	 * @since 1.0.0
	 *
	 * @param  string $column
	 * @param  int    $post_id
	 */
	public function manage_custom_columns( $column, $post_id ) {
		$terms = get_the_terms( $post_id, self::TAXONOMY_TYPE );
		if ( ! $terms || is_wp_error( $terms ) ) {
			return;
		}
		$checked = false;
		$type = '';
		foreach ( $terms as $term ) {
			if( $term->slug == 'enable' ) {
				$checked = true;
			} elseif( $term->slug == 'product' ) {
				$type = 'product';
			} elseif( $term->slug == 'product_archive' ) {
				$type = 'product_archive';
			}
		}
		switch ( $column ) {
			case 'enable_builder':
				$checked = $checked ? 'checked="checked"' : '';
				echo '<div class="motta-builder__toggle-button">';
				echo sprintf(
					'<input type="checkbox" id="builder_enabled_%1$s" class="motta-builder__enabled" %2$s data-nonce="motta_save_enabled_state_%1$s" data-builder-id="%1$s">',
					esc_attr( $post_id ),
					$checked
				);
				echo '<label for="motta_builder_enable'. esc_attr( $post_id ) .'" aria-label="Switch to enable builder"></label>';
				echo '</div>';
				break;

			case 'builder_type':
				if( $type == 'product' ) {
					esc_html_e('Single Product', 'motta-addons');
				} elseif( $type == 'product_archive' ) {
					esc_html_e('Product Archive', 'motta-addons');
				}
				break;
		}
	}

	/**
	 * Sets the enabled meta field to on or off
	 *
	 */
	public static function save_builder_enable() {
		if ( empty( $_POST['post_ID'] ) ) {
			wp_die( -1 );
		}

		$post_ID  = absint( $_POST['post_ID'] );
		$enabled = absint( $_POST['enabled'] ) == 1 ? 'yes' : 'no';
		$terms = get_the_terms( $post_ID, self::TAXONOMY_TYPE );
		$new_builder_type = '';
		$current_builder_type = '';
		if ( $terms && ! is_wp_error( $terms ) ) {
			foreach ( $terms as $term ) {
				if( $term->slug == 'enable') {
					continue;
				}
				$new_builder_type = $term->slug;
				$current_builder_type = $term->slug;
			}

			if( $enabled == 'yes' ) {
				$new_builder_type .= !empty( $new_builder_type ) ? ',' : '';
				$new_builder_type .= 'enable';
			}
		}
		wp_set_post_terms( $post_ID, $new_builder_type, self::TAXONOMY_TYPE );
		$response = self::save_builder_disable($post_ID, $current_builder_type);
		do_action('motta_after_builder_enable', $post_ID);
		wp_send_json_success($response);
	}

	/**
	 * Disable builder when not activated
	 *
	 */
	public static function save_builder_disable($post_ID, $current_builder_type) {
		if( $current_builder_type == 'product' ) {
			$template_id = \Motta\Addons\Elementor\Builder\Single_Product::get_product_template_id();
		} elseif( $current_builder_type == 'product_archive' ) {
			$template_id = \Motta\Addons\Elementor\Builder\Product_Archive::get_product_archive_template_id();
		}
		$response = '';
		if( !empty( $template_id ) && $post_ID != $template_id ) {
			$terms = get_the_terms( $template_id, self::TAXONOMY_TYPE );
			$new_builder_type = '';
			if ( $terms && ! is_wp_error( $terms ) ) {
				foreach ( $terms as $term ) {
					if( $term->slug == 'enable' ) {
						continue;
					}
					$new_builder_type .= !empty( $new_builder_type ) ? ',' : '';
					$new_builder_type .= $term->slug;
				}
				wp_set_post_terms( $template_id, $new_builder_type, self::TAXONOMY_TYPE );
				$response = $template_id;
			}

		}
		return $response;
	}

	public function builder_template_type() {
		if ( ! isset( $_POST ) ) {
			$errormessage = array(
				'message'  => __('Post request dose not found','motta_addons')
			);
			wp_send_json_error( $errormessage );
		}

		if( !(current_user_can('manage_options') || current_user_can('edit_others_posts')) ){
			$errormessage = array(
				'message'  => __('You are unauthorize to create a template','motta_addons')
			);
			wp_send_json_error( $errormessage );
		}

		$nonce = $_POST['nonce'];
		if ( ! wp_verify_nonce( $nonce, 'motta_buider_new_template' ) ) {
			$errormessage = array(
				'message'  => __('Nonce Varification Faild','motta_addons')
			);
			wp_send_json_error( $errormessage );
		}

		$post_title 		= ! empty( $_POST['post_title'] ) ? sanitize_text_field( $_POST['post_title'] ) : '';
		$template_type 		= ! empty( $_POST['template_type'] ) ? sanitize_text_field( $_POST['template_type'] ) : '';
		$set_default 		= ! empty( $_POST['set_default'] ) ? sanitize_text_field( $_POST['set_default'] ) : '';

		$args = [
			'template_type'  => $template_type,
			'post_title'   => $post_title,
			'set_default' => $set_default
		];
		$this->insert_template( $args );
	}

	public function insert_template($args) {
		$args['post_type'] = self::POST_TYPE;
		$args['post_status'] = 'publish';
		$new_post_id = wp_insert_post( $args );

		if( $new_post_id ) {
			$template_type = $args['template_type'];
			if( !empty( $args['set_default'] ) && $args['set_default'] == 'yes' ) {
				$template_type .= ', enable';

				$this->save_builder_disable( $new_post_id, $args['template_type'] );

				do_action('motta_after_builder_enable', $new_post_id);
			}
			wp_set_post_terms( $new_post_id, $template_type, self::TAXONOMY_TYPE );

			$response = array(
				'id'       => $new_post_id,
			);

			wp_send_json_success( $response );
		} else {
			$errormessage = array(
				'message'  => __('There is an error while create a template. Try it again','motta_addons')
			);
			wp_send_json_error( $errormessage );
		}
	}


	/**
	 * Get all WooCommerce screen ids.
	 *
	 * @return array
	 */
	public static function wc_screen_ids($screen_ids) {
		$screen_ids[] = 'motta_builder';

		return $screen_ids;
	}

	public function admin_scripts( $hook ) {
		$screen = get_current_screen();

		if ( in_array( $hook, array('edit.php', 'post-new.php', 'post.php' ) ) && self::POST_TYPE == $screen->post_type ) {
			wp_enqueue_style( 'motta-builder', MOTTA_ADDONS_URL . 'inc/elementor/builder/assets/css/admin.css' );
			wp_enqueue_script( 'motta-builder', MOTTA_ADDONS_URL . 'inc/elementor/builder/assets/js/admin.js', array( 'jquery', 'jquery-ui-sortable' ),'1.0', true );

		}

	}

    public function admin_print_tabs( $views ) {
		$active_class = 'nav-tab-active';
		$current_type = '';
		if( isset( $_GET['template_type'] ) ){
			$active_class = '';
			$current_type = sanitize_key( $_GET['template_type'] );
		}
        ?>
            <div id="motta-template-tabs-wrapper" class="nav-tab-wrapper">
				<div class="motta-menu-area">
					<a class="nav-tab <?php echo esc_attr($active_class); ?>" href="edit.php?post_type=<?php echo esc_attr(self::POST_TYPE); ?>"><?php echo __('All','motta-addons');?></a>
					<?php
						foreach( self::get_template_type() as $tabkey => $tabname ){
							$active_class = ( $current_type == $tabkey ? 'nav-tab-active' : '' );
							echo '<a class="nav-tab ' . $active_class . '" href="edit.php?post_type=' . self::POST_TYPE . '&template_type=' .$tabkey .'">' . $tabname . '</a>';
						}
					?>
				</div>
            </div>
        <?php
		return $views;
    }

	public function query_filter( \WP_Query $query ) {
		if ( ! is_admin() || ! $this->is_current_screen() ) {
			return;
		}

		if( isset( $_GET['template_type'] ) && $_GET['template_type'] != '' && $_GET['template_type'] != 'all') {
			$type = sanitize_key( $_GET['template_type'] ) ;
			$taxquery = array(
				array(
					'taxonomy' => self::TAXONOMY_TYPE,
					'field' => 'slug',
					'terms' => array($type),
					'operator'=> 'IN'
				)
			);
			$query->set('tax_query', $taxquery);
		}
	}


	private function is_current_screen() {
		global $pagenow, $typenow;
		return 'edit.php' === $pagenow && self::POST_TYPE === $typenow;
	}


	public static function get_template_type(){
		return array(
			'product' => esc_html__('Single Product','motta-addons'),
			'product_archive' => esc_html__('Product Archive','motta-addons'),
		);

	}

	/**
	 * Clear popup ids
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public static function delete_builder_cache($post_id) {
		$terms = get_the_terms( $post_id, self::TAXONOMY_TYPE );
		$terms = ! is_wp_error( $terms ) &&  $terms ? wp_list_pluck($terms, 'slug') : '';
		if( $terms && in_array( 'product', $terms ) ) {
			delete_transient( 'motta_wc_product_template_id' );
			\Motta\Addons\Elementor\Builder\Single_Product::get_product_template_id();
		} elseif( $terms && in_array( 'product_archive', $terms ) ) {
			delete_transient( 'motta_wc_product_archive_template_id' );
			\Motta\Addons\Elementor\Builder\Product_Archive::get_product_archive_template_id();
		}
	}

	public function template_type_popup() {
		if( isset( $_GET['post_type'] ) && $_GET['post_type'] == 'motta_builder' ){
			include_once(MOTTA_ADDONS_DIR . 'inc/elementor/builder/templates/template_type_popup.php' );
		}
    }
}