<?php
namespace Motta\Addons\Elementor\Builder\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class Archive_Page_Header extends Widget_Base {
	public function get_name() {
		return 'motta-archive-page-header';
	}

	public function get_title() {
		return esc_html__( '[Motta] Archive Page Header', 'motta-addons' );
	}

	public function get_icon() {
		return 'eicon-header';
	}

	public function get_categories() {
		return ['motta-wc-addons'];
	}

	public function get_keywords() {
		return [ 'woocommerce', 'title', 'archive', 'page header', 'description' ];
	}

	public function get_style_depends() {
		return [ 'motta-product-archive' ];
	}

	protected function register_controls() {
		$this->start_controls_section(
			'section_content',
			[ 'label' => esc_html__( 'Page Header', 'motta-addons' ) ]
		);

		$this->add_control(
			'type',
			[
				'label'     => esc_html__( 'Type', 'motta-addons' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => 'standard',
				'options'   => [
					'standard'   => esc_html__( 'Standard (image and text)', 'motta-addons' ),
					'minimal' => esc_html__( 'Minimal (text only)', 'motta-addons' ),
				],
			]
		);

		$this->add_responsive_control(
			'image',
			[
				'label'   => esc_html__( 'Image', 'motta-addons' ),
				'type'    => Controls_Manager::MEDIA,
				'default' => [
				],
				'selectors' => [
					'{{WRAPPER}} .page-header .page-header__image' => 'background-image: url("{{URL}}");',
				],
				'condition'   => [
					'type' => 'standard',
				],
			]
		);

		$this->add_responsive_control(
			'background_position',
			[
				'label'     => esc_html__( 'Background Position', 'motta-addons' ),
				'type'      => Controls_Manager::SELECT,
				'responsive' => true,
				'options'   => [
					''              => esc_html__( 'Default', 'motta-addons' ),
					'left top'      => esc_html__( 'Left Top', 'motta-addons' ),
					'left center'   => esc_html__( 'Left Center', 'motta-addons' ),
					'left bottom'   => esc_html__( 'Left Bottom', 'motta-addons' ),
					'right top'     => esc_html__( 'Right Top', 'motta-addons' ),
					'right center'  => esc_html__( 'Right Center', 'motta-addons' ),
					'right bottom'  => esc_html__( 'Right Bottom', 'motta-addons' ),
					'center top'    => esc_html__( 'Center Top', 'motta-addons' ),
					'center center' => esc_html__( 'Center Center', 'motta-addons' ),
					'center bottom' => esc_html__( 'Center Bottom', 'motta-addons' ),
				],
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .page-header .page-header__image' => 'background-position: {{VALUE}};',
				],
				'condition'   => [
					'type' => 'standard',
				],
			]
		);

		$this->add_responsive_control(
			'background_repeat',
			[
				'label'     => esc_html__( 'Background Repeat', 'motta-addons' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => 'no-repeat',
				'options'   => [
					'no-repeat' => esc_html__( 'No-repeat', 'motta-addons' ),
					'repeat' 	=> esc_html__( 'Repeat', 'motta-addons' ),
					'repeat-x'  => esc_html__( 'Repeat-x', 'motta-addons' ),
					'repeat-y'  => esc_html__( 'Repeat-y', 'motta-addons' ),
				],
				'selectors' => [
					'{{WRAPPER}} .page-header .page-header__image' => 'background-repeat: {{VALUE}}',
				],
				'condition'   => [
					'type' => 'standard',
				],
			]
		);

		$this->add_responsive_control(
			'background_size',
			[
				'label'     => esc_html__( 'Background Size', 'motta-addons' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => 'cover',
				'options'   => [
					'cover'   => esc_html__( 'Cover', 'motta-addons' ),
					'contain' => esc_html__( 'Contain', 'motta-addons' ),
					'auto'    => esc_html__( 'Auto', 'motta-addons' ),
				],
				'selectors' => [
					'{{WRAPPER}} .page-header .page-header__image' => 'background-size: {{VALUE}}',
				],
				'condition'   => [
					'type' => 'standard',
				],
			]
		);

		$this->add_control(
			'background_overlay',
			[
				'label' => __( 'Background Overlay', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .page-header .page-header__image-overlay' => 'background-color: {{VALUE}}',
				],
				'condition'   => [
					'type' => 'standard',
				],
			]
		);

		$this->add_control(
			'background_color',
			[
				'label' => __( 'Background Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .page-header .page-header__image' => 'background-color: {{VALUE}}',
				],
				'condition'   => [
					'type' => 'standard',
				],
			]
		);

		$this->add_responsive_control(
			'height',
			[
				'label'     => esc_html__( 'Height', 'motta-addons' ),
				'type'      => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'desktop_default' => [
					'size' => 260,
					'unit' => 'px',
				],
				'range'     => [
					'px' => [
						'min' => 10,
						'max' => 600,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .page-header .page-header__content' => 'height: {{SIZE}}{{UNIT}};',
				],
				'condition'   => [
					'type' => 'standard',
				],
			]
		);

		$this->add_responsive_control(
			'align',
			[
				'label' => esc_html__( 'Alignment', 'motta-addons' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => esc_html__( 'Left', 'motta-addons' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'motta-addons' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => esc_html__( 'Right', 'motta-addons' ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'default' => '',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_page_header_style',
			[
				'label' => esc_html__( 'Page Header', 'motta-addons' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'title_heading',
			[
				'label' => esc_html__( 'Title', 'motta-addons' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'title_color',
			[
				'label' => esc_html__( 'Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .page-header .page-header__title' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'title_typography',
				'selector' => '{{WRAPPER}} .page-header .page-header__title',
			]
		);

		$this->add_control(
			'desc_heading',
			[
				'label' => esc_html__( 'Description', 'motta-addons' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'hide_desc',
			[
				'label' => esc_html__( 'Hide', 'motta-addons' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Yes', 'motta-addons' ),
				'label_off' => esc_html__( 'No', 'motta-addons' ),
				'return_value' => 'none',
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .page-header .page-header__description' => 'display: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'desc_spacing',
			[
				'label'     => esc_html__( 'Spacing', 'motta-addons' ),
				'type'      => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 600,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .page-header .page-header__description' => 'margin-top: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'desc_color',
			[
				'label' => esc_html__( 'Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .page-header .page-header__description' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'desc_typography',
				'selector' => '{{WRAPPER}} .page-header .ppage-header__description',
			]
		);

		$this->end_controls_section();
	}

	protected function render() {
		$settings = $this->get_settings_for_display();
		$atts = $this->taxonomy_fields();
		$image_style = $atts && isset( $atts['bg_image'] ) ? 'style=background-image:url(' . $atts['bg_image'] . ')' : '';
		$overlay_style = $atts && isset( $atts['bg_overlay'] ) ? 'style=' . $atts['bg_overlay']  : '';
		$page_header_style = $atts && isset( $atts['color'] ) ? 'style=--motta-text-color:' . isset( $atts['color'] ) : '';

		?>
		<div class="page-header page-header-elementor page-header--products page-header--<?php echo esc_attr( $settings['align'] );?> page-header--<?php echo esc_attr( $settings['type'] ); ?>" <?php echo esc_attr( $page_header_style ) ?>>
			<div class="page-header__content">
				<?php if( $settings['type'] == 'standard' ) { ?>
				<div class="page-header__image" <?php echo esc_attr( $image_style ) ?>>
					<div class="page-header__image-overlay" <?php echo esc_attr( $overlay_style ) ?>></div>
				</div>
				<?php } ?>
				<?php echo apply_filters('motta_page_header_title', '<h1 class="page-header__title">' . get_the_archive_title() . '</h1>');  ?>
				<?php $this->description(); ?>
			</div>
		</div>
		<?php
	}

	public function render_plain_content() {}

	public function get_group_name() {
		return 'woocommerce';
	}

	/**
	 * Get description
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function description() {
		ob_start();
		if( function_exists('is_shop') && is_shop() ) {
			woocommerce_product_archive_description();
		}

		$description = ob_get_clean();

		if ( is_tax() ) {
			$term = get_queried_object();
			if ( $term ) {
				$description = $term->description;
			}
		}

		if( $description ) {
			echo '<div class="page-header__description">'. $description .'</div>';
		}
	}

	/**
	 * Get description
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function taxonomy_fields() {
		$atts = array();
		if ( ! is_product_taxonomy() ) {
			return $atts;
		}

		$term_id  = get_queried_object_id();
		$image_id                                     = absint( get_term_meta( $term_id, 'motta_page_header_bg_id', true ) );
		$motta_page_header_background_overlay         = get_term_meta( $term_id, 'motta_page_header_background_overlay', true );
		$motta_page_header_background_overlay_opacity = get_term_meta( $term_id, 'motta_page_header_background_overlay_opacity', true );

		$motta_page_header_textcolor                  = get_term_meta( $term_id, 'motta_page_header_textcolor', true );
		$motta_page_header_textcolor_custom           = get_term_meta( $term_id, 'motta_page_header_textcolor_custom', true );

		if ( $motta_page_header_textcolor == 'custom' ) {
			$atts['color']          = $motta_page_header_textcolor_custom;
		} elseif ( $motta_page_header_textcolor == 'dark' ) {
			$atts['color']         = '#000';
		} elseif ( $motta_page_header_textcolor == 'light' ) {
			$atts['color']         = '#fff';
		}

		if ( $image_id ) {
			$image             = wp_get_attachment_image_src( $image_id, 'full' );
			if( $image ) {
				$atts['bg_image'] = $image[0];
			}
		}
		$bg_overlay = '';
		if ( $motta_page_header_background_overlay ) {
			$bg_overlay = 'background-color:' . $motta_page_header_background_overlay;
		}

		if( $motta_page_header_background_overlay_opacity ) {
			$bg_overlay .= ';opacity:' . $motta_page_header_background_overlay_opacity;
		}

		$atts['bg_overlay'] = $bg_overlay;

		return $atts;
	}
}
