<?php

namespace Motta\Addons\Elementor\Builder\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Motta\Addons\Elementor\Builder\Current_Query_Renderer;
use Motta\Addons\Elementor\Builder\Products_Renderer;
use Motta\Addons\Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Product Deals Grid widget
 */
class Archive_Products extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'motta-archive-products';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( '[Motta] Archive Products', 'motta-addons' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-archive-posts';
	}

	public function get_categories() {
		return ['motta-wc-addons'];
	}

	public function get_keywords() {
		return [ 'woocommerce', 'shop', 'store', 'product', 'archive' ];
	}

	public function get_script_depends() {
		return [ 'motta-product-archive' ];
	}

	public function get_style_depends() {
		return [ 'motta-product-archive' ];
	}
	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @access protected
	 */
	protected function register_controls() {
		$this->section_content();
	//	$this->section_style();
	}

	protected function section_content() {
		$this->start_controls_section(
			'section_products',
			[ 'label' => esc_html__( 'Products', 'motta-addons' ) ]
		);

		$this->add_control(
			'wc_notice_use_customizer',
			[
				'type' => Controls_Manager::RAW_HTML,
				'raw' => esc_html__( 'To change the Products Archive’s layout, go to Appearance > Customize > WooCommerce > Product Catalog.', 'motta-addons' ),
				'content_classes' => 'elementor-panel-alert elementor-panel-alert-info',
			]
		);

		$this->add_control(
			'grid_border',
			[
				'label'   => esc_html__( 'Border', 'motta-addons' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					''     => esc_html__( 'No Border', 'motta-addons' ),
					'border' => esc_html__( 'Border', 'motta-addons' ),
					'border-bottom' => esc_html__( 'Border Bottom', 'motta-addons' ),
				],
				'default'            => '',
			]
		);

		$this->add_control(
			'grid_border_color',
			[
				'label' => __( 'Border Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} ul.products li.product' => 'border-color: {{VALUE}}',
					'{{WRAPPER}} ul.products li.product:after' => 'background-color: {{VALUE}}',
				],
				'condition'   => [
					'grid_border!' => '',
				],
			]
		);


		$this->add_control(
			'pagination',
			[
				'label'   => esc_html__( 'Pagination', 'motta-addons' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'none'     => esc_html__( 'None', 'motta-addons' ),
					'numberic' => esc_html__( 'Numberic', 'motta-addons' ),
					'infinite' => esc_html__( 'Infinite Scroll', 'motta-addons' ),
					'loadmore' => esc_html__( 'Load More', 'motta-addons' ),
				],
				'default'            => 'numberic',
			]
		);

		$this->end_controls_section();
	}


	/**
	 * Render icon box widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		$settings['paginate'] = $settings['pagination'] == 'numberic' ? true : false;
		add_filter( 'wcboost_wishlist_button_add_text',	array( $this, 'wishlist_button_add_text' ) );
		if (  \Motta\Addons\Elementor\Builder\Helper::is_elementor_editor_mode() ) {
			$shortcode = new Products_Renderer( $settings, 'products' );
		} else {
			$shortcode = new Current_Query_Renderer( $settings, 'current_query' );
		}

		$content = $shortcode->get_content();

		$classes = '';
		if( $settings['grid_border'] != '' ) {
			$classes = 'catalog-grid--has-' . $settings['grid_border'];
		}

		echo '<div class="motta-archive-products ' . esc_attr( $classes ) . '">';
		if ( $content ) {
			$content = str_replace( '<ul class="products', '<ul class="products elementor-grid', $content );

			// PHPCS - Woocommerce output
			echo $content; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		} else {
			echo '<div class="elementor-nothing-found motta-products-nothing-found woocommerce-info">' . esc_html__('No products were found matching your selection.', 'motta-addons') . '</div>';
		}

		$this->pagination($settings);

		echo '</div>';

	}

	/**
	 * Products pagination.
	 */
	public function pagination($settings) {
		$nav_type = $settings['pagination'];
		if( ! in_array( $nav_type, array('infinite', 'loadmore') ) ) {
			return;
		}
		if ( 'loadmore' == $nav_type ) {
			$this->posts_found($nav_type);
		}

		$classes = array(
			'woocommerce-navigation',
			'woocommerce-navigation__catalog',
			'next-posts-navigation',
			'ajax-navigation',
			'ajax-' . $nav_type,
		);
		if ( \Motta\Addons\Elementor\Builder\Helper::is_elementor_editor_mode() && 'infinite' == $nav_type ) {
			$classes[] = 'loading';
		}
		add_filter('next_posts_link_attributes', array( $this, 'posts_link_attributes') );
		echo '<nav class="' . esc_attr( implode( ' ', $classes ) ) . '">';
			if ( \Motta\Addons\Elementor\Builder\Helper::is_elementor_editor_mode() ) {
				echo '<a href="#" class="nav-links motta-button motta-button--bg-color-black motta-button--large">'.esc_html__( 'Load More Products', 'motta' ).'</a>';
			} elseif(get_next_posts_link()) {
				next_posts_link( esc_html__( 'Load More Products', 'motta' ) );
			}

			echo '<div class="motta-pagination--loading">
				<div class="motta-pagination--loading-dots">
					<span></span>
					<span></span>
					<span></span>
					<span></span>
				</div>
				<div class="motta-pagination--loading-text">' . esc_html__( 'Loading more...', 'motta' ) . '</div>
			</div>';
		echo '</nav>';
	}

	/**
	 * Get post found
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function posts_found($nav_type) {
		global $wp_query;

		if ( \Motta\Addons\Elementor\Builder\Helper::is_elementor_editor_mode() ) {
			$post_text = esc_html__( 'items', 'motta' );
			$found_posts = 32;
			$post_count = 16;
		} else {
			if ( $wp_query && $wp_query->found_posts ) {
				$found_posts = $wp_query->found_posts;
				$post_text = $found_posts > 1 ? esc_html__( 'items', 'motta' ) : esc_html__( 'item', 'motta' );
				$post_count = $wp_query->post_count;
			}
		}

		if( ! $found_posts ) {
			return;
		}

		echo sprintf( '<div class="motta-posts-found motta-progress woocommerce-nav-%s">
							<div class="motta-posts-found__inner motta-progress__inner">
							%s
							<span class="current-post"> %s </span>
							%s
							<span class="found-post"> %s </span>
							%s
							<span class="count-bar motta-progress__count-bar"></span>
						</div>
					</div>',
				esc_attr( $nav_type ),
				esc_html__( 'Showing', 'motta' ),
				$post_count,
				esc_html__( 'of', 'motta' ),
				$found_posts,
				$post_text
		);

	}

	public function posts_link_attributes() {
		return 'class="nav-links motta-button motta-button--bg-color-black motta-button--large"';
	}


	/**
	 * Change button text
	 *
	 * @return void
	 */
	public function wishlist_button_add_text() {
		return esc_html__( 'Wishlist', 'motta' );
	}
}
