<?php
namespace Motta\Addons\Elementor\Builder\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Border;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class Product_Add_To_Cart extends Widget_Base {

	use \Motta\Addons\Elementor\Builder\Traits\Product_Id_Trait;

	public function get_name() {
		return 'motta-wc-product-add-to-cart';
	}

	public function get_title() {
		return esc_html__( '[Motta] Add To Cart', 'motta-addons' );
	}

	public function get_icon() {
		return 'eicon-product-add-to-cart';
	}

	public function get_script_depends() {
		return [
			'motta-elementor-single-product'
		];
	}

	public function get_style_depends() {
		return [ 'motta-elementor-single-product' ];
	}

	/**
	 * Get widget categories
	 *
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * @return string Widget categories
	 */
	public function get_categories() {
		return [ 'motta-wc-addons' ];
	}

	public function get_keywords() {
		return [ 'woocommerce', 'shop', 'store', 'cart', 'product', 'button', 'add to cart' ];
	}

	protected function register_controls() {
		$this->start_controls_section(
			'section_atc_button_style',
			[
				'label' => esc_html__( 'Add To Cart Button', 'motta-addons' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'icon_add_to_cart',
			[
				'label' => esc_html__( 'Show Add To Cart Icon', 'motta-addons' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Yes', 'motta-addons' ),
				'label_off' => esc_html__( 'No', 'motta-addons' ),
				'return_value' => 'inline-block',
				'default' => 'inline-block',
				'selectors' => [
					'.woocommerce {{WRAPPER}} .cart .single_add_to_cart_button .single_add_to_cart_button--icon' => 'display: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'ajax_add_to_cart',
			[
				'label' => esc_html__( 'Enable AJAX Add To Cart', 'motta-addons' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Yes', 'motta-addons' ),
				'label_off' => esc_html__( 'No', 'motta-addons' ),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);

		$this->add_responsive_control(
			'button_width',
			[
				'label' => esc_html__( 'Width', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 1,
					],
				],
				'selectors' => [
					'.woocommerce.single-product {{WRAPPER}} .cart .single_add_to_cart_button' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'button_height',
			[
				'label' => esc_html__( 'Height', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 500,
						'step' => 1,
					],
				],
				'selectors' => [
					'.woocommerce {{WRAPPER}} .cart .single_add_to_cart_button' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'button_typography',
				'selector' => '{{WRAPPER}} .cart .single_add_to_cart_button',
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' => 'button_border',
				'selector' => '{{WRAPPER}} .cart .single_add_to_cart_button',
				'exclude' => [ 'color' ],
			]
		);

		$this->add_control(
			'button_border_radius',
			[
				'label' => esc_html__( 'Border Radius', 'motta-addons' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors' => [
					'{{WRAPPER}} .cart .single_add_to_cart_button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'button_padding',
			[
				'label' => esc_html__( 'Padding', 'motta-addons' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em' ],
				'selectors' => [
					'{{WRAPPER}} .cart .single_add_to_cart_button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->start_controls_tabs( 'button_style_tabs' );

		$this->start_controls_tab( 'button_style_normal',
			[
				'label' => esc_html__( 'Normal', 'motta-addons' ),
			]
		);

		$this->add_control(
			'button_text_color',
			[
				'label' => esc_html__( 'Text Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .cart .single_add_to_cart_button' => '--mt-color__primary--light: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'button_bg_color',
			[
				'label' => esc_html__( 'Background Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .cart .single_add_to_cart_button' => '--mt-color__primary: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'button_border_color',
			[
				'label' => esc_html__( 'Border Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .cart .single_add_to_cart_button' => 'border-color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'button_box_shadow_color',
			[
				'label' => esc_html__( 'Box Shadow Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .cart .single_add_to_cart_button' => '--mt-color__primary--box-shadow: {{VALUE}}',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab( 'button_style_hover',
			[
				'label' => esc_html__( 'Hover', 'motta-addons' ),
			]
		);

		$this->add_control(
			'button_text_color_hover',
			[
				'label' => esc_html__( 'Text Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .cart .single_add_to_cart_button:hover' => '--mt-color__primary--light: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'button_bg_color_hover',
			[
				'label' => esc_html__( 'Background Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .cart .single_add_to_cart_button:hover' => '--mt-color__primary--dark: {{VALUE}};--mt-color__primary--darker: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'button_border_color_hover',
			[
				'label' => esc_html__( 'Border Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .cart .single_add_to_cart_button:hover' => 'border-color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'button_box_shadow_hover',
			[
				'label' => esc_html__( 'Box Shadow Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .cart .single_add_to_cart_button:hover' => '--mt-color__primary--box-shadow: {{VALUE}}',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();

		$this->start_controls_section(
			'section_buy_now_button_style',
			[
				'label' => esc_html__( 'Buy Now Button', 'motta-addons' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'buy_now_button_width',
			[
				'label' => esc_html__( 'Width', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 1,
					],
				],
				'selectors' => [
					'.woocommerce {{WRAPPER}} .cart .motta-buy-now-button' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'buy_now_button_height',
			[
				'label' => esc_html__( 'Height', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 500,
						'step' => 1,
					],
				],
				'selectors' => [
					'.woocommerce {{WRAPPER}} .cart .motta-buy-now-button' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'buy_now_button_spacing_top',
			[
				'label' => esc_html__( 'Spacing Top', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} .cart .motta-buy-now-button' => 'margin-top: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'buy_now_button_spacing_left',
			[
				'label' => esc_html__( 'Spacing Left', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', 'em' ],
				'selectors' => [
					'{{WRAPPER}} .cart .motta-buy-now-button' => 'margin-left: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'buy_now_button_typography',
				'selector' => '{{WRAPPER}} .cart .motta-buy-now-button',
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' => 'buy_now_button_border',
				'selector' => '{{WRAPPER}} .cart .motta-buy-now-button',
				'exclude' => [ 'color' ],
			]
		);

		$this->add_control(
			'buy_now_button_border_radius',
			[
				'label' => esc_html__( 'Border Radius', 'motta-addons' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors' => [
					'{{WRAPPER}} .cart .motta-buy-now-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'buy_now_button_padding',
			[
				'label' => esc_html__( 'Padding', 'motta-addons' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em' ],
				'selectors' => [
					'{{WRAPPER}} .cart .motta-buy-now-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->start_controls_tabs( 'buy_now_button_style_tabs' );

		$this->start_controls_tab( 'buy_now_button_style_normal',
			[
				'label' => esc_html__( 'Normal', 'motta-addons' ),
			]
		);

		$this->add_control(
			'buy_now_button_text_color',
			[
				'label' => esc_html__( 'Text Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .cart .motta-buy-now-button' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'buy_now_button_bg_color',
			[
				'label' => esc_html__( 'Background Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .cart .motta-buy-now-button' => 'background-color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'buy_now_button_border_color',
			[
				'label' => esc_html__( 'Border Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .cart .motta-buy-now-button' => 'border-color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'buy_now_button_box_shadow',
			[
				'label' => esc_html__( 'Box Shadow Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .cart .motta-buy-now-button' => '--mt-color__primary--box-shadow: {{VALUE}}',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab( 'buy_now_button_style_hover',
			[
				'label' => esc_html__( 'Hover', 'motta-addons' ),
			]
		);

		$this->add_control(
			'buy_now_button_text_color_hover',
			[
				'label' => esc_html__( 'Text Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .cart .motta-buy-now-button:hover' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'buy_now_button_bg_color_hover',
			[
				'label' => esc_html__( 'Background Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .cart .motta-buy-now-button:hover, {{WRAPPER}} .cart .motta-buy-now-button:focus' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'buy_now_button_border_color_hover',
			[
				'label' => esc_html__( 'Border Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .cart .motta-buy-now-button:hover' => 'border-color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'buy_now_button_box_shadow_hover',
			[
				'label' => esc_html__( 'Box Shadow Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .cart .motta-buy-now-button:hover' => '--mt-color__primary--box-shadow: {{VALUE}}',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();

		$this->start_controls_section(
			'section_atc_quantity_style',
			[
				'label' => esc_html__( 'Quantity', 'motta-addons' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'quantity_spacing_right',
			[
				'label' => esc_html__( 'Spacing Right', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'.woocommerce {{WRAPPER}} .quantity, .woocommerce div.product {{WRAPPER}} .quantity' => 'margin-right: {{SIZE}}{{UNIT}};',
				],

			]
		);

		$this->add_responsive_control(
			'quantity_spacing_bottom',
			[
				'label' => esc_html__( 'Spacing Bottom', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px' ],
				'selectors' => [
					'.woocommerce {{WRAPPER}} .quantity, .woocommerce div.product {{WRAPPER}} .quantity' => 'margin-bottom: {{SIZE}}{{UNIT}};',
					'.woocommerce div.product {{WRAPPER}} .motta-quantity-wrapper' => 'margin-bottom: {{SIZE}}{{UNIT}};',
					'.woocommerce div.product {{WRAPPER}} .motta-quantity-wrapper .quantity' => 'margin-bottom: 0;',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'quantity_typography',
				'selector' => '.woocommerce {{WRAPPER}} .quantity .qty',
			]
		);

		$this->add_control(
			'quantity_border_radius',
			[
				'label' => esc_html__( 'Border Radius', 'motta-addons' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors' => [
					'.woocommerce {{WRAPPER}} .quantity' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'quantity_padding',
			[
				'label' => esc_html__( 'Padding', 'motta-addons' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em' ],
				'selectors' => [
					'.woocommerce {{WRAPPER}} .quantity, .woocommerce div.product {{WRAPPER}} .quantity' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],

			]
		);

		$this->start_controls_tabs( 'quantity_style_tabs',
			[

			]
		);

		$this->start_controls_tab( 'quantity_style_normal',
			[
				'label' => esc_html__( 'Normal', 'motta-addons' ),
			]
		);

		$this->add_control(
			'quantity_text_color',
			[
				'label' => esc_html__( 'Text Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} .quantity .qty' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'quantity_bg_color',
			[
				'label' => esc_html__( 'Background Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} .quantity, .woocommerce {{WRAPPER}} .quantity .qty' => 'background-color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'quantity_border_color',
			[
				'label' => esc_html__( 'Border Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} .quantity' => 'border-color: {{VALUE}}',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab( 'quantity_style_focus',
			[
				'label' => esc_html__( 'Hover', 'motta-addons' ),
			]
		);

		$this->add_control(
			'quantity_text_color_focus',
			[
				'label' => esc_html__( 'Text Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} .quantity:hover .qty' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'quantity_bg_color_focus',
			[
				'label' => esc_html__( 'Background Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} .quantity:hover, .woocommerce {{WRAPPER}} .quantity:hover .qty' => 'background-color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'quantity_border_color_focus',
			[
				'label' => esc_html__( 'Border Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} .quantity:hover' => 'border-color: {{VALUE}}',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_control(
			'heading_qty_label_style',
			[
				'label' => esc_html__( 'Quantity Label', 'motta-addons' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'enable_quantity_label',
			[
				'label' => esc_html__( 'Enable Label', 'motta-addons' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Yes', 'motta-addons' ),
				'label_off' => esc_html__( 'No', 'motta-addons' ),
				'return_value' => 'yes',
				'default' => '',
			]
		);

		$this->add_control(
			'quantity_label',
			[
				'label' => __( 'Title', 'motta-addons' ),
				'type' => Controls_Manager::TEXT,
				'placeholder' => __( 'Quantity:', 'motta' ),
				'condition'   => [
					'enable_quantity_label' => 'yes',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'quantity_label_typography',
				'selector' => '.woocommerce {{WRAPPER}} .quantity__label',
				'condition'   => [
					'enable_quantity_label' => 'yes',
				],
			]
		);

		$this->add_control(
			'quantity_label_color',
			[
				'label' => esc_html__( 'Text Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} .quantity__label' => 'color: {{VALUE}}',
				],
				'condition'   => [
					'enable_quantity_label' => 'yes',
				],
			]
		);

		$this->add_control(
			'quantity_label_number_color',
			[
				'label' => esc_html__( 'Number Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} .quantity__label .quantity__label-number' => 'color: {{VALUE}}',
				],
				'condition'   => [
					'enable_quantity_label' => 'yes',
				],
			]
		);

		$this->add_control(
			'heading_qty_input_style',
			[
				'label' => esc_html__( 'Quantity Input', 'motta-addons' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_responsive_control(
			'quantity_height',
			[
				'label' => esc_html__( 'Height', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px' ],
				'selectors' => [
					'.woocommerce {{WRAPPER}} .quantity .motta-qty-button' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'quantity_width',
			[
				'label' => esc_html__( 'Width', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px' ],
				'selectors' => [
					'.woocommerce {{WRAPPER}} .quantity input.qty' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'heading_qty_icon_style',
			[
				'label' => esc_html__( 'Quantity Icon', 'motta-addons' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'quantity_icon_padding',
			[
				'label' => esc_html__( 'Padding', 'motta-addons' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em' ],
				'selectors' => [
					'.woocommerce {{WRAPPER}} .quantity .motta-qty-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],

			]
		);

		$this->add_control(
			'quantity_icon_size',
			[
				'label' => esc_html__( 'Size', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'.woocommerce {{WRAPPER}} .quantity .motta-qty-button svg' => 'font-size: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'quantity_icon_border_radius',
			[
				'label' => esc_html__( 'Border Radius', 'motta-addons' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors' => [
					'.woocommerce {{WRAPPER}} .quantity .motta-qty-button svg' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->start_controls_tabs( 'quantity_icon_style_tabs',[]);

		$this->start_controls_tab( 'quantity_icon_style_normal',
			[
				'label' => esc_html__( 'Normal', 'motta-addons' ),
			]
		);

		$this->add_control(
			'quantity_icon_text_color',
			[
				'label' => esc_html__( 'Text Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} .quantity .motta-qty-button svg' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'quantity_icon_bg_color',
			[
				'label' => esc_html__( 'Background Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} .quantity .motta-qty-button svg' => 'background-color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'quantity_icon_border_color',
			[
				'label' => esc_html__( 'Border Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} .quantity .motta-qty-button svg' => 'border-color: {{VALUE}}',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab( 'quantity_icon_style_focus',
			[
				'label' => esc_html__( 'Hover', 'motta-addons' ),
			]
		);

		$this->add_control(
			'quantity_icon_text_color_focus',
			[
				'label' => esc_html__( 'Text Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} .quantity .motta-qty-button:hover svg' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'quantity_icon_bg_color_focus',
			[
				'label' => esc_html__( 'Background Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} .quantity .motta-qty-button:hover svg' => 'background-color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'quantity_icon_border_color_focus',
			[
				'label' => esc_html__( 'Border Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} .quantity .motta-qty-button:hover svg' => 'border-color: {{VALUE}}',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();

		$this->start_controls_section(
			'section_atc_variations_style',
			[
				'label' => esc_html__( 'Variations', 'motta-addons' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'variations_spacing',
			[
				'label' => esc_html__( 'Spacing', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'selectors' => [
					'.woocommerce {{WRAPPER}} form.cart .variations' => 'margin-bottom: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->add_control(
			'variations_space_between',
			[
				'label' => esc_html__( 'Space Between', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'selectors' => [
					'.woocommerce {{WRAPPER}} form.cart table.variations tr' => 'padding-top: calc( {{SIZE}}{{UNIT}}/2 ); padding-bottom: calc( {{SIZE}}{{UNIT}}/2 );',
				],
			]
		);

		$this->add_control(
			'heading_variations_label_style',
			[
				'label' => esc_html__( 'Label', 'motta-addons' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'heading_variations_label_between',
			[
				'label' => esc_html__( 'Space', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'selectors' => [
					'.woocommerce {{WRAPPER}} form.cart table.variations .label' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'variations_label_color_focus',
			[
				'label' => esc_html__( 'Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} form.cart table.variations label' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'variations_label_typography',
				'selector' => '.woocommerce {{WRAPPER}} form.cart table.variations label',
			]
		);

		$this->add_control(
			'heading_variations_select_style',
			[
				'label' => esc_html__( 'Select field', 'motta-addons' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'variations_select_color',
			[
				'label' => esc_html__( 'Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} form.cart table.variations td.value select' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'variations_select_bg_color',
			[
				'label' => esc_html__( 'Background Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} form.cart table.variations td.value select, .woocommerce {{WRAPPER}} form.cart table.variations td.value:before' => 'background-color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'variations_select_border_color',
			[
				'label' => esc_html__( 'Border Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} form.cart table.variations td.value select, .woocommerce {{WRAPPER}} form.cart table.variations td.value:before' => 'border: 1px solid {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'variations_select_typography',
				'selector' => '.woocommerce {{WRAPPER}} form.cart table.variations td.value select, .woocommerce div.product.elementor{{WRAPPER}} form.cart table.variations td.value:before',
			]
		);

		$this->add_control(
			'variations_select_border_radius',
			[
				'label' => esc_html__( 'Border Radius', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors' => [
					'.woocommerce {{WRAPPER}} form.cart table.variations td.value select, .woocommerce {{WRAPPER}} form.cart table.variations td.value:before' => 'border-radius: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->add_control(
			'heading_variations_type_color_style',
			[
				'label' => esc_html__( 'Color field', 'motta-addons' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'variations_type_color_width',
			[
				'label' => esc_html__( 'Width', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 200,
						'step' => 1,
					],
				],
				'selectors' => [
					'.woocommerce {{WRAPPER}} .wcboost-variation-swatches--color .wcboost-variation-swatches__item' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'variations_type_color_height',
			[
				'label' => esc_html__( 'Height', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 200,
						'step' => 1,
					],
				],
				'selectors' => [
					'.woocommerce {{WRAPPER}} .wcboost-variation-swatches--color .wcboost-variation-swatches__item' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'variations_type_color_border_color',
			[
				'label' => esc_html__( 'Border Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} .wcboost-variation-swatches--color .wcboost-variation-swatches__item:hover' => 'border-color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'variations_type_color_border_radius',
			[
				'label' => esc_html__( 'Border Radius', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors' => [
					'.woocommerce {{WRAPPER}} .wcboost-variation-swatches--color .wcboost-variation-swatches__item .wcboost-variation-swatches__name' => 'border-radius: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->add_control(
			'heading_variations_type_image_style',
			[
				'label' => esc_html__( 'Image field', 'motta-addons' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'variations_type_image_width',
			[
				'label' => esc_html__( 'Width', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 200,
						'step' => 1,
					],
				],
				'selectors' => [
					'.woocommerce {{WRAPPER}} .wcboost-variation-swatches--image .wcboost-variation-swatches__item' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'variations_type_image_border_color',
			[
				'label' => esc_html__( 'Border Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} .wcboost-variation-swatches--image .wcboost-variation-swatches__item:hover' => 'border-color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'variations_type_image_border_radius',
			[
				'label' => esc_html__( 'Border Radius', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors' => [
					'.woocommerce {{WRAPPER}} .wcboost-variation-swatches--image .wcboost-variation-swatches__item img' => 'border-radius: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->add_control(
			'heading_variations_type_label_style',
			[
				'label' => esc_html__( 'Label field', 'motta-addons' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'variations_type_label_width',
			[
				'label' => esc_html__( 'Width', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 200,
						'step' => 1,
					],
				],
				'selectors' => [
					'.woocommerce {{WRAPPER}} .wcboost-variation-swatches--label .wcboost-variation-swatches__item' => 'min-width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'variations_type_label_height',
			[
				'label' => esc_html__( 'Height', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 200,
						'step' => 1,
					],
				],
				'selectors' => [
					'.woocommerce {{WRAPPER}} .wcboost-variation-swatches--label .wcboost-variation-swatches__item' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'variations_type_label_color',
			[
				'label' => esc_html__( 'Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} .wcboost-variation-swatches--label .wcboost-variation-swatches__item' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'variations_type_label_border_color',
			[
				'label' => esc_html__( 'Border Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} .wcboost-variation-swatches--label .wcboost-variation-swatches__item' => 'border-color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'variations_type_label_border_color_hover',
			[
				'label' => esc_html__( 'Border Color hover', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} .wcboost-variation-swatches--label .wcboost-variation-swatches__item:hover' => 'border-color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'variations_type_label_typography',
				'selector' => '.woocommerce {{WRAPPER}} .wcboost-variation-swatches--label .wcboost-variation-swatches__item',
			]
		);

		$this->add_control(
			'variations_type_label_border_radius',
			[
				'label' => esc_html__( 'Border Radius', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors' => [
					'.woocommerce {{WRAPPER}} .wcboost-variation-swatches--label .wcboost-variation-swatches__item' => 'border-radius: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->add_control(
			'heading_variations_type_button_style',
			[
				'label' => esc_html__( 'Button field', 'motta-addons' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'vvariations_type_button_width',
			[
				'label' => esc_html__( 'Width', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 200,
						'step' => 1,
					],
				],
				'selectors' => [
					'.woocommerce {{WRAPPER}} .wcboost-variation-swatches--button .wcboost-variation-swatches__item' => 'min-width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'variations_type_button_height',
			[
				'label' => esc_html__( 'Height', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 200,
						'step' => 1,
					],
				],
				'selectors' => [
					'.woocommerce {{WRAPPER}} .wcboost-variation-swatches--button .wcboost-variation-swatches__item' => 'min-height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'variations_type_button_color',
			[
				'label' => esc_html__( 'Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} .wcboost-variation-swatches--button .wcboost-variation-swatches__item' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'variations_type_button_border_color',
			[
				'label' => esc_html__( 'Border Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} .wcboost-variation-swatches--button .wcboost-variation-swatches__item' => 'border-color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'variations_type_button_border_color_hover',
			[
				'label' => esc_html__( 'Border Color Hover', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} .wcboost-variation-swatches--button .wcboost-variation-swatches__item:hover' => 'border-color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'variations_type_button_typography',
				'selector' => '.woocommerce {{WRAPPER}} .wcboost-variation-swatches--button .wcboost-variation-swatches__item',
			]
		);

		$this->add_control(
			'variations_type_button_border_radius',
			[
				'label' => esc_html__( 'Border Radius', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors' => [
					'.woocommerce {{WRAPPER}} .wcboost-variation-swatches--button .wcboost-variation-swatches__item' => 'border-radius: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->end_controls_section();
		$this->start_controls_section(
			'section_stock_style',
			[
				'label' => esc_html__( 'Stock', 'motta-addons' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'product_stock',
			[
				'label' => esc_html__( 'Enable Stock', 'motta-addons' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Yes', 'motta-addons' ),
				'label_off' => esc_html__( 'No', 'motta-addons' ),
				'return_value' => 'yes',
				'default' => '',
			]
		);
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'text_typography',
				'selector' => '{{WRAPPER}} .stock',
			]
		);
		$this->add_control(
			'text_color',
			[
				'label' => esc_html__( 'Text Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .stock' => 'color: {{VALUE}}',
				],
			]
		);
		$this->add_control(
			'in_stock_color',
			[
				'label' => esc_html__( 'In Stock Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .stock.in-stock' => 'color: {{VALUE}}',
				],
			]
		);
		$this->add_control(
			'out_stock_color',
			[
				'label' => esc_html__( 'Out Stock Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .stock.out-stock' => 'color: {{VALUE}}',
				],
			]
		);
		$this->end_controls_section();
	}

	protected function render() {
		global $product;

		$product = $this->get_product();

		if ( ! $product ) {
			return;
		}
		$settings = $this->get_settings_for_display();
		$class = $settings['ajax_add_to_cart'] == 'yes' ? 'product-add-to-cart-ajax' : '';

		if($settings['enable_quantity_label'] == 'yes' ) {
			add_action( 'woocommerce_before_add_to_cart_quantity', array( $this, 'quantity_label' ) );
		}

		if($settings['product_stock'] == 'yes' ) {
			add_action( 'woocommerce_before_add_to_cart_quantity', array( $this, 'open_quantity_wrapper' ) );
			add_action( 'woocommerce_after_add_to_cart_quantity', array( $this, 'stock' ) );
			add_action( 'woocommerce_after_add_to_cart_quantity', array( $this, 'close_quantity_wrapper' ) );
		}

		?>

		<div class="motta-elementor-add-to-cart <?php echo esc_attr($class); ?>">
			<?php
				woocommerce_template_single_add_to_cart();
				if($settings['product_stock'] == 'yes' ) {
					if ( ! $product->is_in_stock() && $product->get_type() == 'simple' )  {
						$this->stock();
					}
				}
			?>
		</div>

		<?php
		if($settings['product_stock'] == 'yes' ) {
			remove_action( 'woocommerce_before_add_to_cart_quantity', array( $this, 'open_quantity_wrapper' ) );
			remove_action( 'woocommerce_after_add_to_cart_quantity', array( $this, 'stock' ) );
			remove_action( 'woocommerce_after_add_to_cart_quantity', array( $this, 'close_quantity_wrapper' ) );
		}

		if($settings['enable_quantity_label'] == 'yes' ) {
			remove_action( 'woocommerce_before_add_to_cart_quantity', array( $this, 'quantity_label' ) );
		}

	}

	public function render_plain_content() {}

	public function get_group_name() {
		return 'woocommerce';
	}

	public function stock() {
		global $product;

		if ( ! $product ) {
			return;
		}
		if( $product->is_type( 'grouped' ) ) {
			return;
		}

		echo wc_get_stock_html( $product );
	}

		/**
	 * Open quantity wrapper
     *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function open_quantity_wrapper() {
		echo '<div class="motta-quantity-wrapper variations-attribute-change">';
	}

	/**
	 * Close quantity wrapper
     *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function close_quantity_wrapper() {
		echo '</div>';
	}


	/**
	 * Quantity label
	 *
	 * @return void
	 */
	public function quantity_label() {
		global $product;
		if( $product->is_sold_individually() ) {
			return;
		}
		$settings = $this->get_settings_for_display();
		$label = $settings['quantity_label'];
		$label = empty( $label ) ? esc_html__( 'Quantity:', 'motta') : $label;
		echo '<div class="quantity__label">' . $label . '<span class="quantity__label-number"></span></div>';
	}
}
