<?php
namespace Motta\Addons\Elementor\Builder\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Repeater;
use Elementor\Icons_Manager;
use Elementor\Group_Control_Typography;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 *
 * @since 1.0.0
 */
class Product_Badge extends Widget_Base {

	use \Motta\Addons\Elementor\Builder\Traits\Product_Id_Trait;

	/**
	 * Get widget name.
	 *
	 * Retrieve social icons widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'motta-wc-product-badge';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve social icons widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( '[Motta] Product Badge', 'motta-addons' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve social icons widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-flash';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return ['motta-wc-addons'];
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'product', 'badge', 'motta-addons' ];
	}

	public function get_script_depends() {
		return [
			'motta-countdown',
			'motta-elementor-single-product'
		];
	}

	public function get_style_depends() {
		return [ 'motta-elementor-single-product' ];
	}

	/**
	 * Register social icons widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 3.1.0
	 * @access protected
	 */
	protected function register_controls() {
		$this->start_controls_section(
			'section_product_badge',
			[
				'label' => esc_html__( 'Product Badge', 'motta-addons' ),
			]
		);

		$this->add_control(
			'heading_product_sale',
			[
				'label' => esc_html__( 'Product Sale', 'motta-addons' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'product_sale_bg_color',
			[
				'label' => __( 'Background Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woocommerce-badges .onsale' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'product_sale_color',
			[
				'label' => __( 'Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woocommerce-badges .onsale' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'heading_product_featured',
			[
				'label' => esc_html__( 'Product Featured', 'motta-addons' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'product_featured_bg_color',
			[
				'label' => __( 'Background Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woocommerce-badge.featured' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'product_featured_color',
			[
				'label' => __( 'Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woocommerce-badge.featured' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'heading_product_new',
			[
				'label' => esc_html__( 'Product New', 'motta-addons' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'product_new_bg_color',
			[
				'label' => __( 'Background Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woocommerce-badge.new' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'product_new_color',
			[
				'label' => __( 'Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woocommerce-badge.new' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'heading_product_deal',
			[
				'label' => esc_html__( 'Product Deal', 'motta-addons' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'product_deal_type',
			[
				'label'      => esc_html__( 'Type', 'motta-addons' ),
				'type'       => Controls_Manager::SELECT,
				'options'    => [
					''     => esc_html__( 'Text', 'motta-addons' ),
					'countdown'  => esc_html__( 'Countdown', 'motta-addons' ),
				],
				'default'    => '',
			]
		);

		$this->add_control(
			'ajax_add_to_cart',
			[
				'label' => esc_html__( 'Hide Limited Time Offer Text', 'motta-addons' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Yes', 'motta-addons' ),
				'label_off' => esc_html__( 'No', 'motta-addons' ),
				'return_value' => 'none',
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} .motta-single-product-sale > .motta-single-product-sale__text' => 'display: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'product_deal_bg_image',
			[
				'label'   => esc_html__( 'Background Image', 'motta-addons' ),
				'type'    => Controls_Manager::MEDIA,
				'default' => [],
				'selectors' => [
					'{{WRAPPER}} .motta-single-product-sale' => 'background-image: url("{{URL}}");',
				],
				'condition' => [
					'product_deal_type' => 'countdown',
				],
			]
		);

		$this->add_control(
			'product_deal_bg_color',
			[
				'label' => __( 'Background Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .motta-single-product-sale' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'product_deal_type' => 'countdown',
				],
			]
		);

		$this->add_control(
			'product_deal_color',
			[
				'label' => __( 'Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .motta-single-product-sale' => 'color: {{VALUE}};',
				],
				'condition' => [
					'product_deal_type' => 'countdown',
				],
			]
		);

		$this->add_control(
			'product_deal_text_color',
			[
				'label' => __( 'Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .woocommerce-badges .woocommerce-badge--text' => 'color: {{VALUE}};',
				],
				'condition' => [
					'product_deal_type' => '',
				],
			]
		);


		$this->end_controls_section();
	}


	/**
	 * Render social icons widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		global $product;

		$product = $this->get_product();

		if ( ! $product ) {
			return;
		}

		if( ! class_exists('\Motta\WooCommerce\Badges') ) {
			return;
		}
		$settings = $this->get_settings_for_display();
		$is_product =  \Motta\Addons\Elementor\Builder\Helper::is_elementor_editor_mode() ? is_singular( 'motta_builder' ) : is_singular('product');
		$args = array(
			'is_product' => $is_product
		);
		$sale_date = \Motta\WooCommerce\Badges::get_date_on_sale_to($product, $args);
		if ( $product->is_on_sale() && $settings['product_deal_type'] == 'countdown' && ! empty( $sale_date ) ) {
			echo \Motta\WooCommerce\Badges::get_product_sale_countdown();
			if (  \Motta\Addons\Elementor\Builder\Helper::is_elementor_editor_mode() ) {
				?>
				<script>
					jQuery('.motta-countdown').motta_countdown();
				</script>
				<?php
			}
		} else {
			$args = array(
				'badges_soldout'   => true,
				'badges_soldout_text' => esc_html__( 'Out Of Stock', 'motta' ),
				'badges_sale'   => true,
				'badges_sale_text' => esc_html__( 'Sale', 'motta' ),
				'badges_new'   => true,
				'badges_new_text' => esc_html__( 'New', 'motta' ),
				'badges_featured'      => true,
				'badges_featured_text' => esc_html__( 'Hot', 'motta' ),
				'is_product' => $is_product
			);
			$badges = \Motta\WooCommerce\Badges::get_badges($product, $args);
			if( ! $badges ) {
				return;
			}
			?>
			<div class="woocommerce-badges variations-attribute-change">
				<?php echo implode( '', $badges ); ?>
			</div>
		<?php
		}
	}

}
