<?php
namespace Motta\Addons\Elementor\Builder\Widgets;

use Motta\Addons\Elementor\Base\Products_Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Image_Size;
use \Motta\Addons\Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Pricing Table widget
 */
class Products_List extends Products_Widget_Base {

	use \Motta\Addons\Elementor\Builder\Traits\Product_Id_Trait;
	/**
	 * Retrieve the widget name.
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'motta-wc-products-list';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( '[Motta] Products List', 'motta-addons' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-post-list';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return ['motta-wc-addons'];
	}

	public function get_style_depends() {
		return [ 'motta-elementor-single-product' ];
	}

	/**
	 * Get widget keywords.
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords() {
		return [ 'products list', 'products', 'list', 'woocommerce', 'motta-addons' ];
	}

	/**
	 * Register the widget controls.
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @access protected
	 */
	protected function register_controls() {
		$this->section_content();
		$this->section_style();
	}

	// Tab Content
	protected function section_content() {
		$this->start_controls_section(
			'section_product_list',
			[ 'label' => __( 'Products List', 'motta-addons' ) ]
		);

		$this->add_control(
			'title',
			[
				'label'       => esc_html__( 'Title', 'motta-addons' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => esc_html__( 'This is heading', 'motta-addons' ),
				'label_block' => true,
			]
		);

		$this->add_control(
			'limit',
			[
				'label'   => esc_html__( 'Total Products', 'motta-addons' ),
				'type'    => Controls_Manager::NUMBER,
				'default' => 5,
				'min'     => 1,
				'max'     => 50,
				'step'    => 1,
			]
		);

		$this->add_control(
			'products',
			[
				'label'   => esc_html__( 'Type', 'motta-addons' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'recent_products'       => __( 'Recent Products', 'motta-addons' ),
					'featured_products'     => __( 'Featured Products', 'motta-addons' ),
					'sale_products'         => __( 'Sale Products', 'motta-addons' ),
					'best_selling_products' => __( 'Best Selling Products', 'motta-addons' ),
					'top_rated_products'    => __( 'Top Rated Products', 'motta-addons' ),
				],
				'default' => 'recent_products',
				'toggle'  => false,
			]
		);

		$this->add_control(
			'orderby',
			[
				'label'     => esc_html__( 'Order By', 'motta-addons' ),
				'type'      => Controls_Manager::SELECT,
				'options'   => [
					'menu_order' => __( 'Menu Order', 'motta-addons' ),
					'date'       => __( 'Date', 'motta-addons' ),
					'title'      => __( 'Title', 'motta-addons' ),
					'price'      => __( 'Price', 'motta-addons' ),
				],
				'condition' => [
					'products'            => [ 'top_rated', 'sale', 'featured' ],
				],
				'default'   => 'date',
			]
		);

		$this->add_control(
			'order',
			[
				'label'     => esc_html__( 'Order', 'motta-addons' ),
				'type'      => Controls_Manager::SELECT,
				'options'   => [
					''     => esc_html__( 'Default', 'motta-addons' ),
					'asc'  => esc_html__( 'Ascending', 'motta-addons' ),
					'desc' => esc_html__( 'Descending', 'motta-addons' ),
				],
				'condition' => [
					'products'            => [ 'top_rated', 'sale', 'featured' ],
				],
				'default'   => '',
			]
		);

		$this->add_control(
			'taxonomy_condition',
			[
				'label'     => esc_html__( 'Condition', 'motta-addons' ),
				'type'      => Controls_Manager::SELECT,
				'options'   => [
					''     => esc_html__( 'Default', 'motta-addons' ),
					'single'  => esc_html__( 'For Single Product', 'motta-addons' ),
				],
				'default'   => '',
			]
		);

		$this->add_control(
			'product_cats',
			[
				'label'       => esc_html__( 'Product Categories', 'motta-addons' ),
				'placeholder' => esc_html__( 'Click here and start typing...', 'motta-addons' ),
				'type'        => 'motta-autocomplete',
				'default'     => '',
				'label_block' => true,
				'multiple'    => true,
				'source'      => 'product_cat',
				'sortable'    => true,
				'condition' => [
					'taxonomy_condition'  => '',
				],

			]
		);

		$this->add_control(
			'product_tags',
			[
				'label'       => esc_html__( 'Product Tags', 'motta-addons' ),
				'placeholder' => esc_html__( 'Click here and start typing...', 'motta-addons' ),
				'type'        => 'motta-autocomplete',
				'default'     => '',
				'label_block' => true,
				'multiple'    => true,
				'source'      => 'product_tag',
				'sortable'    => true,
				'condition' => [
					'taxonomy_condition'  => '',
				],

			]
		);

		$this->add_control(
			'product_brands',
			[
				'label'       => esc_html__( 'Product Brands', 'motta-addons' ),
				'placeholder' => esc_html__( 'Click here and start typing...', 'motta-addons' ),
				'type'        => 'motta-autocomplete',
				'default'     => '',
				'label_block' => true,
				'multiple'    => true,
				'source'      => 'product_brand',
				'sortable'    => true,
				'condition' => [
					'taxonomy_condition'  => '',
				],
			]
		);

		$this->add_control(
			'product_cat',
			[
				'label'       => esc_html__( 'Same Product Category', 'motta-addons' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', 'motta-addons' ),
				'label_off'    => esc_html__( 'Hide', 'motta-addons' ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'condition' => [
					'taxonomy_condition'  => 'single',
				],

			]
		);

		$this->add_control(
			'product_tag',
			[
				'label'       => esc_html__( 'Same Product Tag', 'motta-addons' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', 'motta-addons' ),
				'label_off'    => esc_html__( 'Hide', 'motta-addons' ),
				'return_value' => 'yes',
				'default'      => '',
				'default'      => 'yes',
				'condition' => [
					'taxonomy_condition'  => 'single',
				],

			]
		);

		$this->add_control(
			'product_brand',
			[
				'label'       => esc_html__( 'Same Product Brand', 'motta-addons' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', 'motta-addons' ),
				'label_off'    => esc_html__( 'Hide', 'motta-addons' ),
				'return_value' => 'yes',
				'default'      => '',
				'default'      => 'yes',
				'condition' => [
					'taxonomy_condition' => 'single',
				],

			]
		);

		$this->add_control(
			'product_outofstock',
			[
				'label'        => esc_html__( 'Show Out Of Stock Products', 'motta-addons' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', 'motta-addons' ),
				'label_off'    => esc_html__( 'Hide', 'motta-addons' ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);

		$this->end_controls_section();
	}

	// Tab Style
	protected function section_style() {
		$this->start_controls_section(
			'section_style_products_list',
			[
				'label' => esc_html__( 'Products List', 'motta-addons' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'products_list_padding',
			[
				'label' => esc_html__( 'Padding', 'motta-addons' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em' ],
				'selectors' => [
					'{{WRAPPER}} .motta-wc-products-list' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' => 'products_list_border',
				'selector' => '{{WRAPPER}} .motta-wc-products-list',
			]
		);

		$this->add_control(
			'title_style_divider',
			[
				'label' => __( 'Title', 'motta-addons' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'title_color',
			[
				'label' => esc_html__( 'Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} .motta-wc-products-list h2' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'title_spacing',
			[
				'label'     => __( 'Spacing', 'motta-addons' ),
				'type'      => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'range'     => [
					'px' => [
						'max' => 1000,
						'min' => 0,
					],
					'%' => [
						'max' => 100,
						'min' => 0,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .motta-wc-products-list h2' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);


		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'title_typography',
				'selector' => '.woocommerce {{WRAPPER}} .motta-wc-products-list h2',
			]
		);

		$this->add_control(
			'product_item_style_divider',
			[
				'label' => __( 'Product Item', 'motta-addons' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_responsive_control(
			'product_item_spacing',
			[
				'label'     => __( 'Spacing', 'motta-addons' ),
				'type'      => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'range'     => [
					'px' => [
						'max' => 1000,
						'min' => 0,
					],
					'%' => [
						'max' => 100,
						'min' => 0,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .motta-products-list__item' => '--motta-product-list-item__spacing: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'product_item_border_color',
			[
				'label' => esc_html__( 'Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'.woocommerce {{WRAPPER}} .motta-products-list__item' => '--motta-product-list-item__border-color: {{VALUE}}',
				],
			]
		);


		$this->add_control(
			'image_style_divider',
			[
				'label' => __( 'Image', 'motta-addons' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_group_control(
			Group_Control_Image_Size::get_type(),
			[
				'name'      => 'image',
				'default'   => 'shop_thumbnail',
				'separator' => 'none',
			]
		);

		$this->add_responsive_control(
			'image_width',
			[
				'label'     => __( 'Image Width', 'motta-addons' ),
				'type'      => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'range'     => [
					'px' => [
						'max' => 1000,
						'min' => 0,
					],
					'%' => [
						'max' => 100,
						'min' => 0,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .motta-products-list__image' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'image_spacing',
			[
				'label'     => __( 'Spacing', 'motta-addons' ),
				'type'      => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'range'     => [
					'px' => [
						'max' => 1000,
						'min' => 0,
					],
					'%' => [
						'max' => 100,
						'min' => 0,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .motta-products-list__image' => 'margin-right: {{SIZE}}{{UNIT}};',
				],
			]
		);


		$this->end_controls_section();
	}

	/**
	 * Render widget output on the frontend.
	 * Written in PHP and used to generate the final HTML.
	 */
	protected function render() {
		global $product;
		$product = $this->get_product();

		$settings = $this->get_settings_for_display();

		$this->add_render_attribute( 'wrapper', 'class', [
			'motta-wc-products-list',
		] );

		?>
		<div <?php echo $this->get_render_attribute_string( 'wrapper' ); ?>>
			<?php
			$attr = [
				'type' 				=> $settings['products'],
				'limit'    			=> $settings['limit'],
				'orderby'    		=> $settings['orderby'],
				'order'    			=> $settings['order'],
			];

			if( is_singular('product') || is_singular('motta_builder') ) {
				$attr['post__not_in'] = $product->get_id();
			}

			if( $settings['taxonomy_condition'] == 'single' ) {
				if( is_singular('product') || is_singular('motta_builder') ) {
					if ( $settings['product_cat'] == 'yes' ) {
						$terms = wp_get_post_terms( $product->get_id(), 'product_cat', array( "fields" => "slugs", 'orderby' => 'parent', 'order'   => 'DESC', ) );
						if ( $terms ) {
							$attr['category'] = $terms[0];
						}
					}

					if ( $settings['product_tag'] == 'yes' ) {
						$terms = wp_get_post_terms( $product->get_id(), 'product_tag', array( "fields" => "slugs", 'orderby' => 'parent', 'order'   => 'DESC', ) );
						if ( $terms ) {
							$attr['tag'] = $terms[0];
						}
					}

					if ( $settings['product_brand'] == 'yes' ) {
						$terms = wp_get_post_terms( $product->get_id(), 'product_brand', array( "fields" => "slugs", 'orderby' => 'parent', 'order'   => 'DESC', ) );
						if ( $terms ) {
							$attr['product_brands'] = $terms[0];
						}
					}
				}
			} else {
				$attr['category'] = $settings['product_cats'];
				$attr['tag'] = $settings['product_tags'];
				$attr['product_brands'] = $settings['product_brands'];
			}

			if ( isset( $settings['product_outofstock'] ) && empty( $settings['product_outofstock'] ) ) {
				$attr['product_outofstock'] = $settings['product_outofstock'];
			}

			$product_ids = Utils::products_shortcode( $attr );

			$product_ids = ! empty($product_ids) ? $product_ids['ids'] : 0;

			if( $settings['title'] ) {
				echo sprintf('<h2>%s</h2>', $settings['title']);
			}
			?>
			<div class="motta-products-list__items">
				<?php
				foreach( $product_ids as $id ) {
					$_product = wc_get_product( $id );
					$price    = $_product->get_price_html();
					$image_id = get_post_thumbnail_id( $id );
					$image    = Group_Control_Image_Size::get_attachment_image_src( $image_id, 'image', $settings );
					?>
					<a class="motta-products-list__item" href="<?php echo get_permalink( $id ); ?>">
						<?php echo ! empty( $image ) ? sprintf( '<img src="%s" alt="%s" class="motta-products-list__image">', $image, esc_html( get_the_title( $id ) ) ) : '';; ?>
						<div class="motta-products-list__content">
							<div class="motta-products-list__title"><?php echo esc_html( get_the_title( $id ) ); ?></div>
							<?php $this->get_ratings( $_product ); ?>
							<div class="motta-products-list__price price"><?php echo wp_kses_post( $price ); ?></div>
						</div>
					</a>
				<?php } ?>
			</div>
		</div>
		<?php
	}

	/**
	 * Get ratings
	 *
	 * @return void
	 */
	protected function get_ratings( $product ) {
		if ( $product->get_rating_count() ) {
			echo '<div class="motta-products-list__rating">';
				echo wc_get_rating_html( $product->get_average_rating() );
				if( intval( $product->get_review_count() ) > 0 ) {
					?><span class="review-count">(<?php echo esc_html( $product->get_review_count() ); ?>)</span><?php
				}
			echo '</div>';
		}
	}

}