<?php
namespace Motta\Addons\Elementor\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Utils;
use Elementor\Icons_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Border;

use Motta\Addons\Helper;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Icon Box widget
 */
class Icon_Box_List extends Widget_Base {
	use \Motta\Addons\Elementor\Widgets\Traits\Button_Trait;

	/**
	 * Retrieve the widget name.
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'motta-icon-box-list';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return __( '[Motta] Icons Box List', 'motta-addons' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-icon-box';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return ['motta-addons'];
	}

	/**
	 * Get widget keywords.
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @return array Widget keywords.
	 */
   	public function get_keywords() {
	   return [ 'icon box list', 'icon', 'icons', 'box', 'carousel', 'motta-addons' ];
   	}


	/**
	 * Register the widget controls.
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @access protected
	 */
	protected function register_controls() {
		$this->content_sections();
		$this->style_sections();
	}

	protected function content_sections() {
		$this->start_controls_section(
			'section_icon',
			[ 'label' => __( 'Icon Box', 'motta-addons' ) ]
		);

		$this->add_control(
			'layout',
			[
				'label' => esc_html__( 'Layout', 'motta-addons' ),
				'type' => Controls_Manager::CHOOSE,
				'default' => 'horizontal',
				'options' => [
					'horizontal' => [
						'title' => esc_html__( 'Horizontal', 'motta-addons' ),
						'icon' => 'eicon-ellipsis-h',
					],
					'vertical' => [
						'title' => esc_html__( 'Vertical', 'motta-addons' ),
						'icon' => 'eicon-ellipsis-v',
					],
				],
				'prefix_class' => 'motta-icon-list--',
				'toggle' => false,
			]
		);
		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'icon_type',
			[
				'label' => __( 'Icon Type', 'motta-addons' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'icon' => __( 'Icon', 'motta-addons' ),
					'image' => __( 'Image', 'motta-addons' ),
					'external' => __( 'External', 'motta-addons' ),
				],
				'default' => 'icon',
			]
		);

		$repeater->add_control(
			'icon',
			[
				'label' => __( 'Icon', 'motta-addons' ),
				'type' => Controls_Manager::ICONS,
				'default' => [
					'value' => 'fa fa-star',
					'library' => 'fa-solid',
				],
				'condition' => [
					'icon_type' => 'icon',
				],
			]
		);

		$repeater->add_control(
			'image',
			[
				'label' => __( 'Choose Image', 'motta-addons' ),
				'type' => Controls_Manager::MEDIA,
				'dynamic' => [
					'active' => true,
				],
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],
				'condition' => [
					'icon_type' => 'image',
				],
			]
		);

		$repeater->add_control(
			'icon_url',
			[
				'label' => __( 'External Icon URL', 'motta-addons' ),
				'type' => Controls_Manager::TEXT,
				'dynamic' => [
					'active' => true,
				],
				'condition' => [
					'icon_type' => 'external',
				],
			]
		);

		$repeater->add_control(
			'title',
			[
				'label' => __( 'Title', 'motta-addons' ),
				'type' => Controls_Manager::TEXT,
				'default' => __( 'This is the heading', 'motta-addons' ),
				'placeholder' => __( 'Enter your title', 'motta-addons' ),
				'label_block' => true,
				'separator' => 'before',
			]
		);

		$repeater->add_control(
			'button_text',
			[
				'label' => __('Button Text', 'motta-addons'),
				'type' => Controls_Manager::TEXT,
				'default' => '',
			]
		);

		$repeater->add_control(
			'button_link',
			[
				'label' => __('Button Link', 'motta-addons'),
				'type' => Controls_Manager::URL,
				'default' => [
					'url' => '',
				],
			]
		);

		$repeater->add_control(
			'item_icon_background_color',
			[
				'label' => __( 'Background Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} {{CURRENT_ITEM}} .motta-icon-box-list__icon' => 'background-color: {{VALUE}}',
				],
			]
		);

		$repeater->add_control(
			'item_icon_color',
			[
				'label' => __( 'Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} {{CURRENT_ITEM}} .motta-icon-box-list__icon' => 'color: {{VALUE}}',
				],
			]
		);

		$repeater->add_control(
			'item_icon_border_color',
			[
				'label' => __( 'Border Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} {{CURRENT_ITEM}} .motta-icon-box-list__icon' => 'border-color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'icons',
			[
				'label'       => __( 'Icons Box', 'motta-addons' ),
				'type'        => Controls_Manager::REPEATER,
				'fields'      => $repeater->get_controls(),
				'title_field' => '{{{ title }}}',
				'default' => [
					[
						'title'    => __( 'Name #1', 'motta-addons' ),
					],
					[
						'title'    => __( 'Name #2', 'motta-addons' ),
					],

				],
				'separator' => 'before',
			]
		);

		$this->add_control(
			'title_size',
			[
				'label' => __( 'Title HTML Tag', 'motta-addons' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'h1' => 'H1',
					'h2' => 'H2',
					'h3' => 'H3',
					'h4' => 'H4',
					'h5' => 'H5',
					'h6' => 'H6',
					'div' => 'div',
					'span' => 'span',
					'p' => 'p',
				],
				'default' => 'h6',
			]
		);

		$this->end_controls_section();

	}


	protected function style_sections() {
		$this->content_style_sections();
		$this->icon_style_sections();
	}

	protected function icon_style_sections() {
		// Style Icon
		$this->start_controls_section(
			'section_style_icon',
			[
				'label'     => __( 'Icon', 'motta-addons' ),
				'tab'       => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'icon_color',
			[
				'label' => __( 'Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .motta-icon-box-list__icon' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'icon_spacing',
			[
				'label' => __( 'Spacing', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 200,
					]
				],
				'default' => [],
				'selectors' => [
					'{{WRAPPER}} .motta-icon-box-list__icon' => '--motta-icon-box-list__icon-spacing: {{SIZE}}{{UNIT}};',

				],
			]
		);

		$this->add_responsive_control(
			'icon_size',
			[
				'label' => __( 'Size', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 500,
					]
				],
				'default' => [],
				'selectors' => [
					'{{WRAPPER}} .icon-type-icon .motta-icon-box-list__icon' => 'font-size: {{SIZE}}{{UNIT}}',
					'{{WRAPPER}} .icon-type-external .motta-icon-box-list__icon' => 'width: {{SIZE}}{{UNIT}}',
					'{{WRAPPER}} .icon-type-image .motta-icon-box-list__icon' => 'width: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->add_responsive_control(
			'icon_padding',
			[
				'label'      => esc_html__( 'Padding', 'motta-addons' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .motta-icon-box-list__icon' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' => 'icon_border',
				'label' => esc_html__( 'Border', 'motta-addons' ),
				'selector' => '{{WRAPPER}} .motta-icon-box-list__icon',
			]
		);

		$this->add_control(
			'icon_border_radius',
			[
				'label'      => __( 'Border Radius', 'motta-addons' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .motta-icon-box-list__icon' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);


		$this->end_controls_section();
	}



	protected function content_style_sections() {
		// Content style
		$this->start_controls_section(
			'section_style_content',
			[
				'label' => __( 'Content', 'motta-addons' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'content_padding',
			[
				'label'      => esc_html__( 'Padding', 'motta-addons' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .motta-icon-box-list' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name' => 'content_item_border',
				'label' => esc_html__( 'Border', 'motta-addons' ),
				'selector' => '{{WRAPPER}} .motta-icon-box-list',
			]
		);

		$this->add_control(
			'content_border_radius',
			[
				'label'      => __( 'Border Radius', 'motta-addons' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .motta-icon-box-list' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'item_style_heading',
			[
				'label' => __( 'Item', 'motta-addons' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'item_border_color',
			[
				'label' => __( 'Border Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .motta-icon-box-list__item, {{WRAPPER}} .motta-icon-box-list__item:after' => 'border-color: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'item_space_between',
			[
				'label' => __( 'Spacing', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 200,
					],
				],
				'default' => [],
				'selectors'  => [
					'{{WRAPPER}} .motta-icon-box-list__item' => '--motta-icon-box-list__item-spacing:{{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'title_style_heading',
			[
				'label' => __( 'Title', 'motta-addons' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'title_color',
			[
				'label' => __( 'Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .motta-icon-box-list__title' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'title_typography',
				'selector' => '{{WRAPPER}} .motta-icon-box-list__title',
			]
		);

		$this->add_control(
			'button_style_heading',
			[
				'label' => __( 'Button', 'motta-addons' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_responsive_control(
			'button_space_between',
			[
				'label' => __( 'Spacing', 'motta-addons' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 200,
					],
				],
				'default' => [],
				'selectors'  => [
					'{{WRAPPER}} .motta-icon-box-list__item' => '--motta-icon-box-list__button-spacing:{{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'button_color',
			[
				'label' => __( 'Color', 'motta-addons' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .motta-icon-box-list__button' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'button_typography',
				'selector' => '{{WRAPPER}} .motta-icon-box-list__button',
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render icon box widget output on the frontend.
	 * Written in PHP and used to generate the final HTML.
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$classes = [
			'motta-icon-box-list',
		];

		$this->add_render_attribute( 'wrapper', 'class', $classes );

		$output = array();

		foreach( $settings['icons'] as $index => $slide ) {
			$wrapper_key = $this->get_repeater_setting_key( 'slide_wrapper', 'icons', $index );
			$icon_key = $this->get_repeater_setting_key( 'icon', 'icons', $index );
			$title_key = $this->get_repeater_setting_key( 'title', 'icons', $index );

			$this->add_render_attribute( $wrapper_key, 'class', [ 'motta-icon-box-list__item', 'icon-type-' . $slide['icon_type'], 'elementor-repeater-item-' . $slide['_id'] ] );
			$this->add_render_attribute( $icon_key, 'class', [ 'motta-icon-box-list__icon'] );
			$this->add_render_attribute( $title_key, 'class', [ 'motta-icon-box-list__title'] );

			$this->add_inline_editing_attributes( 'title', 'none' );
			$this->add_inline_editing_attributes( 'description', 'basic' );

			$icon_exist = true;
			$icon = '';

			if ( 'image' == $slide['icon_type'] ) {
				$icon_exist = ! empty($slide['image']) ? true : false;
			} elseif ( 'external' == $slide['icon_type'] ) {
				$icon_exist = ! empty($slide['icon_url']) ? true : false;
			} else {
				$icon_exist = ! empty($slide['icon']) && ! empty($slide['icon']['value']) ? true : false;
			}

			if ( $icon_exist ) {
				$icon = '<div ' . $this->get_render_attribute_string( $icon_key ) . '>';

				if ( 'image' == $slide['icon_type'] ) {
					$icon .= $slide['image'] ? sprintf( '<img alt="%s" src="%s">', esc_attr( $slide['title'] ), esc_url( $slide['image']['url'] ) ) : '';
				} elseif ( 'external' == $slide['icon_type'] ) {
					$icon .= $slide['icon_url'] ? sprintf( '<img alt="%s" src="%s">', esc_attr( $slide['title'] ), esc_url( $slide['icon_url'] ) ) : '';
				} else {
					$icon .= '<span class="motta-svg-icon">';
					$icon .= Icons_Manager::try_get_icon_html( $slide['icon'], [ 'aria-hidden' => 'true' ] );
					$icon .= '</span>';
				}

				$icon .= '</div>';
			}

			$title 		= $slide['title'] ? '<'. Utils::validate_html_tag( $settings['title_size'] ) . ' ' . $this->get_render_attribute_string( $title_key ) . '>' . wp_kses_post( $slide['title'] ) . '</' . Utils::validate_html_tag( $settings['title_size'] ) .'>' : '';

			$output[] = sprintf(
				'<div %s>
					%s
					%s
					%s
				</div>',
				$this->get_render_attribute_string( $wrapper_key ),
				$icon,
				$title,
				! empty( $slide['primary_button_link']['url'] ) ? $this->icon_box_render_button( $slide, $index ) : '',
			);
		}

		echo sprintf(
			'<div %s>
				%s
			</div>',
			$this->get_render_attribute_string( 'wrapper' ),
			implode( '', $output )
		);
	}


	protected function icon_box_render_button ( $slide, $index ) {
		$settings = $this->get_settings_for_display();

		$link_key = 'primary_button_' . $index;

		$this->add_render_attribute( $link_key , 'class', [ 'motta-button', 'motta-button--subtle',  'motta-icon-box-list__button' ] );

		if ( ! empty( $slide['button_link']['url'] ) ) {
			$this->add_link_attributes( $link_key, $slide['button_link'] );
		}

		return sprintf(
			'<a %s><span class="motta-button__text">%s</span></a>',
			$this->get_render_attribute_string( $link_key ),
			$slide['button_text'],
		);
	}
}