<?php

namespace Motta\Addons\Modules;

/**
 * Class for shortcodes.
 */
class Shortcodes {
	/**
	 * Instance
	 *
	 * @var $instance
	 */
	protected static $instance = null;

	/**
	 * Initiator
	 *
	 * @since 1.0.0
	 * @return object
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Instantiate the object.
	 *
	 * @since 1.0.0
	 *
	 * @return void
	 */
	public function __construct() {
		$this->init();
	}

	/**
	 * Init shortcodes
	 */
	public static function init() {
		add_shortcode( 'motta_year', array( __CLASS__, 'year' ) );
		add_shortcode( 'motta_more', array( __CLASS__, 'motta_more_shortcode' ) );
		add_shortcode( 'motta_last_revised', array( __CLASS__, 'last_revised_date' ) );
		add_action( 'pre_get_posts', array(__CLASS__, 'product_search_by_sku_hook' ) );
	}

	/**
	 * Display current year
	 *
	 * @return void
	 */
	public static function year() {
		return date('Y');
	}

	/**
	 * Show more
	 *
	 * @return void
	 */
	public static function motta_more_shortcode( $args, $content ) {
		$default = array(
			'more'   => esc_html__( 'Show More', 'motta-addons' ),
			'less'   => esc_html__( 'Show Less', 'motta-addons' )
		);

		$atts = shortcode_atts( $default, $args );
   		$content = do_shortcode( $content );

		return sprintf(
			'<div class="motta-more" data-settings="%s">
				<div class="motta-more__content">%s</div>
				<button class="motta-more__button motta-button--subtle">%s</button>
			</div>',
			htmlspecialchars( json_encode( $default ) ),
			$content,
			$atts['more']
		);
	}

	/**
	 * Display last update date
	 *
	 * @return void
	 */
	public static function last_revised_date() {
		return get_the_date();
	}

	/**
	 * Add filter to posts clauses to support searching products by sku.
	 *
	 * @param object $query
	 */
	public static function product_search_by_sku_hook( $query ) {
		if ( ( is_admin() && ! defined( 'DOING_AJAX' ) ) || ! $query->is_search() || ! in_array( 'product', (array) $query->get( 'post_type' ) ) ) {
			return;
		}

		add_filter( 'posts_clauses', array(__CLASS__,'product_search_by_sku_query_clauses') );
	}

	/**
	 * Modify the product search query clauses to support searching by sku.
	 *
	 * @todo Support searching in product_variation
	 * @param array $clauses
	 * @return array
	 */
	public static function product_search_by_sku_query_clauses( $clauses ) {
		global $wpdb;

		// Double check because we can't remove filter.
		if (
			! get_query_var( 's' )
			|| ! in_array( 'product', (array) get_query_var( 'post_type' ) )
		) {
			return $clauses;
		}

		$join    = $clauses['join'];
		$where   = $clauses['where'];
		$groupby = $clauses['groupby'];

		// Use the wc_product_meta_lookup, for a better performance.
		if ( $wpdb->wc_product_meta_lookup ) {
			if( ! str_contains( $join, "LEFT JOIN {$wpdb->wc_product_meta_lookup} wc_product_meta_lookup" ) ) {
				$join .= " LEFT JOIN {$wpdb->wc_product_meta_lookup} wc_product_meta_lookup ON $wpdb->posts.ID = wc_product_meta_lookup.product_id ";
			}

			$where = preg_replace(
				"/\(\s*{$wpdb->posts}.post_title\s+LIKE\s*(\'[^\']+\')\s*\)/",
				"({$wpdb->posts}.post_title LIKE $1) OR (wc_product_meta_lookup.sku LIKE $1)", $where );
		}

		// GROUP BY: product id; to avoid duplication.
		$id_group = "{$wpdb->posts}.ID";

		if ( ! strlen( trim( $groupby ) ) ) {
			$groupby = $id_group;
		} elseif ( ! preg_match( "/$id_group/", $groupby ) ) {
			$groupby = $groupby . ', ' . $id_group;
		}

		$clauses['join']    = $join;
		$clauses['where']   = $where;
		$clauses['groupby'] = $groupby;

		return $clauses;
	}
}