const fs = require( 'jsdoc/fs' );
const env = require( 'jsdoc/env' );
const path = require( 'jsdoc/path' );

/** @module publish */

// RegExp used to match the replacement. The groups are respectively: start marker, replacable content, end marker.
const defaultReplacementRegex = /(<woo-tracking-jsdoc(?:\s[^>]*)?>)([\s\S]*)(<\/woo-tracking-jsdoc.*>)/;

const disclaimerStart = `<!---
Everything below will be automatically generated by \`woo-tracking-jsdoc\`.
Do not edit it manually!
-->`;
const disclaimerEnd = `<!---
End of \`woo-tracking-jsdoc\`-generated content.
-->
`;

/**
 * Creates MD link to the line of code, where the symbol was defined.
 *
 * @param {Object} symbol JSDoc symbol object.
 * @param {string} readmeDir The absolute path to the directory of README file.
 */
function getLineLink( symbol, readmeDir ) {
	const localLocation =
		path.relative(
			readmeDir,
			path.join( symbol.meta.path, symbol.meta.filename )
		) +
		'#L' +
		symbol.meta.lineno;
	return `[\`${ symbol.name }\`](${ localLocation })`;
}

/**
 * Generate documentation output.
 *
 * @param {Object} data - A TaffyDB collection representing
 *                       all the symbols documented in your code.
 */
exports.publish = function ( data ) {
	const {
		conf: {
			templates: {
				'woo-tracking-jsdoc': {
					path: readmePath = 'TRACKING.md',
					replacement,
				} = {},
			} = {},
		} = {},
		pwd,
	} = env;

	const readmeDir = path.join( pwd, path.dirname( readmePath ) );
	const replacementRegex = replacement
		? new RegExp( replacement )
		: defaultReplacementRegex;

	let mdResult = '';

	data( { kind: 'event' } )
		.order( 'name' )
		.each( ( symbol ) => {
			// Build the event title with the link to its source.
			mdResult += `\n### ${ getLineLink( symbol, readmeDir ) }\n`;
			// description
			mdResult += ( symbol.description || '' ) + '\n';
			// Build properites table.
			if ( symbol.properties ) {
				mdResult += `#### Properties
|   |   |   |
|---|---|---|\n`;
				symbol.properties.forEach( ( property ) => {
					// Escape `|` for markdown table.
					const type = property.type.parsedType.typeExpression.replace(
						/\|/g,
						'\\|'
					);
					const description = property.description.replace(
						/\|/g,
						'\\|'
					);
					mdResult += `\`${
						property.name
					}\` | \`${ type }\` | ${ description.replace(
						/\s*\n\s*/g,
						' <br> '
					) }\n`;
				} );
			}

			// Find all places that fires the event.
			const emitters = new Map();
			// TaffyDB#has is buggy https://github.com/typicaljoe/taffydb/issues/19, so let's filter it manually.
			data( { fires: { isArray: true } } ).each( ( emitter ) => {
				const firesCurrent = emitter.fires.filter(
					( fires ) => fires.name === 'event:' + symbol.name
				);
				if ( firesCurrent.length ) {
					emitters.set( emitter, firesCurrent );
				}
			} );
			if ( emitters.size ) {
				mdResult += `#### Emitters\n`;
				emitters.forEach( ( fires, emitter ) => {
					mdResult += '- ' + getLineLink( emitter, readmeDir );
					if ( fires.length > 1 ) {
						mdResult +=
							`\n` +
							fires
								.map(
									( evt ) => `	- ${ evt.description || '' }`
								)
								.join( '\n' );
					} else if ( fires[ 0 ] && fires[ 0 ].description ) {
						mdResult += ' ' + fires[ 0 ].description;
					}
					mdResult += '\n';
				} );
			}
		} );

	let readme = fs.readFileSync( readmePath, 'utf8' );
	// Replace the marker with generated content.
	readme = readme.replace(
		replacementRegex,
		`$1\n${ disclaimerStart }\n${ mdResult }\n${ disclaimerEnd }$3`
	);
	fs.writeFileSync( readmePath, readme, 'utf8' );
	return mdResult;
};
